#   TSIPDevice.py is part of PyTSIPChat - a software written in Python to 
#   communicate with the Trimble Products that use the proprietary TSIP protocol.
#
#   Copyright (C) 2014 Fernando P. Hauscarriaga <fernandoph@iar.unlp.edu.ar>
#
#   PyTSIPChat is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   PyTSIPChat is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with PyTSIPChat.  If not, see <http://www.gnu.org/licenses/>.

from copy import deepcopy
import struct
from TSIPPacket import TSIPPacket
from PacketsDefinitions import *

# "Constants"
EMPTY = 0x0
FULL  = 0x1
DLE1  = 0x2
DATA  = 0x3
DLE2  = 0x4

# Packet delimiter special characters
DLE     = 0x10
ETX     = 0x03
SUPERPK = 0x8F

class TSIPDevice(object):
    def __init__(self, bufsize = None):
        self.RXBuffer = []
        self.TXBuffer = []
        # Ports devices (Serial Class)
        self.ports = []
        
        if bufsize:
            self.bufferSize = bufsize
        else:
            self.bufferSize = 256
        
        #"Private" attributes (Python doesnt really support private 
        # attributes, we distinguish them in sake of clarity
        # Two lists of the TSIPPacket_Class to Transmit and Receive packets 
        # to the TSIPDevice, simulating a Queue
        # We must "insert in the front" of the list, so when we use the pop
        # method, we are popping the first element that came in (FIFO)

#================================= Methods =====================================
# Getters
    def getRXBuffer(self):
        return RXBuffer
        
    def getTXBuffer(self):
        return self.TXBuffer
        
    def getPorts(self):
        return self.ports
        
    def getNumberOfPorts(self):
        return self.ports.__len__()

    def getBufferSize(self):
        return self.bufferSize

# Setters
    def addPort(self, aConfiguredSerialPort):
        self.ports.append(aConfiguredSerialPort)
    
    def addAPacketToTXBuffer(self, aTSIPPacket):
        self.TXBuffer.insert(0,  aTSIPPacket)
        
    def addAPacketToRXBuffer(self, aTSIPPacket):
        self.RXBuffer.insert(0, aTSIPPacket)

# Functionality
    def openPorts(self):
        for port in self.getPorts():
            port.open()
            port.flushInput()
            port.flushOutput()

    def closePorts(self):
        for port in self.getPorts():
            port.close()

    # Read a single port and return a non iterpreted TSIP Packet
    # (chars only)
    def readPort(self, aPort):
        
        c = aPort.read(self.getBufferSize())
        i = 0

        if len(c) == 0:
            return None
        else:
            packets = []
        
        while i < len(c):

            pkid   = None
            data   = ''
            status = EMPTY

            while True and i < len(c):
                if c[i] == '':
                    break
        
                if status == DLE1:
                    if ord(c[i]) == 0 or ord(c[i]) == DLE or ord(c[i]) == ETX:
                        status = EMPTY
                    else:
                        status = DATA
                        pkid   = ord(c[i])
                        data  += c[i]
                
                elif status == DATA:
                    if ord(c[i]) == DLE:
                        status = DLE2
                    else:
                        data  += c[i]
        
                elif status == DLE2:
                    if ord(c[i]) == DLE:
                        status = DATA
                        data  += c[i]
                    elif ord(c[i]) == ETX:
                        status = FULL
                    else:
                        # Error: start new report packet
                        status = DLE1
                        data  += c[i]
                elif status == FULL or status == EMPTY:
                    #       pass
                    #   else:
                    if ord(c[i]) != DLE:
                        status = EMPTY
                    else:
                        status = DLE1
                    
                if status == FULL:
                    # We are done
                    packets.append(data)
                    break;

                i += 1

        #return data
        return packets

    def writePort(self, aPort, aTSIPPacket):
        #<DLE> <id> <data string bytes> <DLE> <ETX>
        data  = '\x10' # DLE

        tmpstr = ''

        i = 0

        for pos in aTSIPPacket.getPacket()[1][1:]:
            c = struct.pack('>'+pos, aTSIPPacket.getPacket()[2][i][1])
            tmpstr += c
            if ord(c) == DLE:
                tmpstr += c

            i += 1

        data += tmpstr
        
        data += '\x10' #DLE
        data += '\x03' #ETX
        
        aPort.write(data)

        return data

    # Read all ports
    def readPorts(self):
        packets = []

        for port in self.ports:
            # Read port
            pks = self.readPort(port)

            if pks != None:
                for pk in pks:
                    # A fresh new TSIPPacket object
                    packet = TSIPPacket()
                    # Interpret bytes
                    if (packet.interpret(pk) != None):
                        packets.append(deepcopy(packet))
                    # Cleanup, object copied
                    del packet

        return packets

    def getAvalCmds(self):
        cmdlist = []

        for key in TSIPSUPERCMD.keys():
            cmdlist.append("0x8E-%02X - %s" % (key, TSIPSUPERCMD[key][0]))

        return cmdlist
