/** 
 *   Camera Control for STAR I.
 *
 *   (c) Copyright 2004 Federico Bareilles <fede@iar.unlp.edu.ar>,
 *   Instituto Argentino de Radio Astronomia (IAR).
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License as
 *   published by the Free Software Foundation; either version 2 of
 *   the License, or (at your option) any later version.
 *     
 *   The author does NOT admit liability nor provide warranty for any
 *   of this software. This material is provided "AS-IS" in the hope
 *   that it may be useful for others.
 *
 **/
#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <time.h>
#include <getopt.h>
#include <unistd.h>
#include <limits.h>

#include <readline/readline.h>
#include <readline/history.h>

#include "my_stdio.h"
#include "my_math.h"
#include "camera.h"
#include "fits_io.h"
#include "version.h"
#include "camera_tools.h"
#include "camera_xpa.h"

int cam;

/* The names of functions that actually do the manipulation. */
int com_help  __P((char *));
int com_quit  __P((char *));
int com_test  __P((char *));
int com_time  __P((char *));
int com_temperature  __P((char *));
int com_etime  __P((char *));
int com_exposure  __P((char *));
int com_get  __P((char *));
int com_focus  __P((char *));
int com_shade  __P((char *));
int com_clear  __P((char *));
int com_gain  __P((char *));
int com_dark  __P((char *));
int com_bias  __P((char *));

int com_temp_reg __P((char *));

int com_nroi __P((char *));
int com_roi_expose __P((char *));
int com_roi __P((char *));

int com_box_x0 __P((char *));
int com_box_y0 __P((char *));
int com_box_x_len __P((char *));
int com_box_y_len __P((char *));

int com_box_get __P((char *));
int com_box_set __P((char *));

int com_mouse_get __P((char *));
int com_system __P((char *));



/* A structure which contains information on the commands this program
   can understand. */

typedef struct {
	char *name;		/* User printable name of the function. */
	rl_icpfunc_t *func;	/* Function to call to do the job. */
	char *param;            /* Parameters for this function.  */
	char *doc;		/* Documentation for this function.  */
} COMMAND;

COMMAND commands[] = {
	{ "test", com_test,"", "[!]" },
	{ "time", com_time,"", "Set date & time on camera controller" },
	{ "temp", com_temperature,"", "Get current CCD temperature" },
	{ "etime", com_etime, "[time]", "Get/Set exposure time (sec)" },
	{ "treg", com_temp_reg,"[on|off]", "Get/Set teperature regulator" },
	{ "expose", com_exposure,"", "Exposure" },
	{ "dark", com_dark,"", "Dark integration" },
	{ "bias", com_bias,"", "Bias integration" },

	{ "get", com_get,"{filename.fits}", 
	  "Get current image from the camera controller" },
	{ "focus", com_focus,"", "100x100 pixel focus integration" },
	{ "shade", com_shade,"", "Test video display memory" },
	{ "clear", com_clear,"", "Clear video display memory" },
	{ "gain", com_gain,"[1|4]", "Get/Set system gain factor" },

	{ "nroi", com_nroi,"", "Get/Set ROI number" },
	{ "roiexp", com_roi_expose,"", "Expose ROI" },
	{ "roi", com_roi, "[x0 y0 xlen ylen]", "Get/Set ROI" },

	{ "gpointer", com_mouse_get, "", 
	  "Get current pointer location and pixel value"},
//	{ "x0", com_box_x0,"", "Get/Set X origin of box" },
//	{ "y0", com_box_y0,"", "Get/Set Y origin of box" },
//	{ "xlen", com_box_x_len,"", "Get/Set length X of box" },
//	{ "ylen", com_box_y_len,"", "Get/Set length Y of box" },

//	{ "gbox", com_box_get,"", "Get box" },
//	{ "sbox", com_box_set,"", "Set box" },
	
	
//	{ "help", com_help, "[cmd]", "Display this text" },
	{ "?", com_help, "[cmd]", "Synonym for `help'" },
//	{ "quit", com_quit, "", "Quit ..." },
	{ "q", com_quit, "", "Quit :-)" },
	{ "!", com_system, "", "Shell exec" },
	{ (char *)NULL, (rl_icpfunc_t *)NULL, (char *) NULL, (char *)NULL }
};

/* When non-zero, this means the user is done using this program. */
int done = 0;


pid_t my_exec(char *command)
{
	pid_t pid;
	
	if (command == 0)
		return 1;
	pid = fork();
	if (pid == -1)
		return -1;
	if (pid == 0) {
		char *argv[4];
		argv[0] = "sh";
		argv[1] = "-c";
		argv[2] = command;
		argv[3] = 0;
		if( execve("/bin/sh", argv, environ) == -1){
			perror("execve");
			exit(1); /* ojo: execve no retorna si tiene exito. */
		}
		exit(127);
	}
	return pid;
}



/* Generator function for command completion.  STATE lets us
   know whether to start from scratch; without any state
   (i.e. STATE == 0), then we start at the top of the list. */
char *command_generator (char *text, int state)
{ 
	static int list_index, len;
	char *name;

	/* If this is a new word to complete, initialize now.  This
	   includes saving the length of TEXT for efficiency, and
	   initializing the index variable to 0. */
	if (!state) {
		list_index = 0;
		len = strlen (text);
	}

	/* Return the next name which partially matches from the
	   command list. */
	while ((name = commands[list_index].name)) {
		list_index++;
		if (strncmp (name, text, len) == 0)
			return (strdup(name));
	}

	/* If no names matched, then return NULL. */
	return ((char *)NULL);
}


char **fileman_completion (const char *text, int start, int end)
{
	char **matches;

	matches = (char **)NULL;

	/* If this word is at the start of the line, then it is a
	   command to complete.  Otherwise it is the name of a file in
	   the current directory.  */
	if (start == 0)
	  matches = rl_completion_matches (text, 
					   (rl_compentry_func_t *) command_generator);
	
	return (matches);
}


/* Tell the GNU Readline library how to complete.  We want to try to
   complete on command names if this is the first word in the line, or
   on filenames if not. */
void initialize_readline (void)
{
	/* Allow conditional parsing of the ~/.inputrc file. */
	rl_readline_name = "rotator";
	
	/* Tell the completer that we want a crack first. */
	rl_attempted_completion_function = fileman_completion;
}


/* Look up NAME as the name of a command, and return a pointer to that
   command.  Return a NULL pointer if NAME isn't a command name. */
COMMAND *find_command (char *name)
{
	int i;

	for (i = 0; commands[i].name; i++)
		if (strcmp (name, commands[i].name) == 0)
			return (&commands[i]);

	return ((COMMAND *)NULL);
}


/* Execute a command line. */
int execute_line (char *line)
{
  int i;
	COMMAND *command;
	char *word;

	/* Isolate the command word. */
	i = 0;
	while (line[i] && WHITESPACE(line[i]))	i++;
	word = line + i;

	while (line[i] && !WHITESPACE (line[i])) i++;

	if (line[i])
		line[i++] = '\0';

	command = find_command (word);

	if (!command) {
		fprintf (stderr, "%s: No such command for stari.\n", word);
		return (-1);
	}

	/* Get argument to command, if any. */
	while (WHITESPACE (line[i])) i++;

	word = line + i;

	/* Call the function. */
	return ((*(command->func)) (word));
}


/* The user wishes to quit using this program.  Just set DONE
   non-zero. */
int com_quit (char *arg)
{
	done = 1;
	return (0);
}

/* Print out help for ARG, or for all of the commands if ARG is not
   present. */
int com_help (char *arg)
{
	int i, j;
	int printed = 0;
	int cmdlen;
	const int param_legend = 22;

	for (i = 0; commands[i].name; i++) {
		if (!*arg || (strcmp (arg, commands[i].name) == 0)) {
			cmdlen = strlen(commands[i].name) + 
				strlen(commands[i].param) + 1;
			printf ("%s %s", commands[i].name, 
				commands[i].param);
			if (cmdlen >= param_legend) {
				printf("\n");
				cmdlen = 0;
			}
			for(j = cmdlen; j < param_legend; j++)
				printf(" ");
			printf ("%s.\n", commands[i].doc);

			printed++;
		}
	}

	if (!printed) {
		printf ("No commands match `%s'.  Possibilties are:\n", arg);

		for (i = 0; commands[i].name; i++) {
			/* Print in six columns. */
			if (printed == 6) {
				printed = 0;
				printf ("\n");
			}
			
			printf ("%s\t", commands[i].name);
			printed++;
		}

		if (printed)
			printf ("\n");
	}
	return (0);
}



int command_main(void)
{
	char *line, *s;

	initialize_readline ();	/* Bind our completer. */

	/* Loop reading and executing lines until the user quits. */
	for ( ; done == 0; ) {
		line = readline ("stari> ");
		if (!line)
			break;
		s = stripwhite (line);

		if (*s)
		{/*
			if (s[0] == '!' && s[1] != ' ') {
				s[0] = ' ';
				}*/
			add_history (s);
			execute_line (s);
		}
		
		free (line);
	}
	return (0);
}



int com_time (char *arg)
{
	time_t now;

	time(&now);
	cc_set_date_time(cam, now);

	return 0;
}

int com_temperature (char *arg)
{ 
 	int t;

	cc_get_temperature(cam, &t);
	printf(" Currente camera head temperature is %d\n", t);

	return 0;
}


int com_etime (char *arg)
{
	int etime;

        if (arg[0] == 0) {
		cc_get_exposure_time(cam, &etime);
		printf(" current exposure time is %.1f\n", 
		       (double) etime / 10.0);

                return 0;
        }
	etime = (int) my_round(atof(arg) * 10.0);
	cc_set_exposure_time(cam, etime);

	return 0;
}


int com_gain (char *arg)
{
	int gain;

        if (arg[0] == 0) {
		cc_get_gain(cam, &gain);
		if ( gain == GAIN_HIGH ) gain = 4;
		else gain = 1;
                printf(" current gain is %d\n", gain);
                return 0;
        }

	gain = atoi(arg);
	switch(gain){
	case 1:
		gain = GAIN_LOW;
		break;
	case 4:
		gain = GAIN_HIGH;
		break;
	default:
		printf(" gain: bad argument %s\n", arg);
		return -1;
	}
	cc_set_gain(cam, gain);

	return 0;
}


int com_temp_reg (char *arg)
{
	int reg;
	char sta[4];

        if (arg[0] == 0) {
		cc_get_temp_regulation(cam, &reg);
		if ( reg == TEMP_REG_ON ) sprintf(sta,"on");
		else sprintf(sta,"off");
                printf(" temperature regulator is %s\n", sta);
                return 0;
        }

	if( strncmp(arg, "on", 2) == 0)
		reg = TEMP_REG_ON;
	else if(strncmp(arg, "off", 3) == 0) 
		reg = TEMP_REG_OFF;
	else {
		printf(" tempreg: bad argument %s\n", arg);
		return -1;
	}
	cc_set_temp_regulation(cam, reg);

	return 0;
}


int com_exposure (char *arg)
{
	cc_act_exposure(cam);
	return 0;
}


int com_dark (char *arg)
{
	cc_act_dark(cam);
	return 0;
}


int com_bias (char *arg)
{
	cc_act_bias(cam);
	return 0;
}


int com_get (char *arg)
{
	char filename[_POSIX_PATH_MAX];
	time_t t;
	int good_image_flag = 0;

	if (arg[0] == 0) {
		time(&t);
		sprintf(filename,"im%10ld.fits", t); 
		
	}else{
		sprintf(filename,"%s", arg);
	}

	switch (ct_read_save_file(cam, filename)) {
	case CT_IM_OK:
		good_image_flag = 1;
		printf("ok.\n");
		break;
	case CT_IM_SETUP:
		printf(" Error reading image setup.\n");
		break;
	case CT_IM_DIM:
		printf(" Invalid dimension.\n");
		break;
	case CT_IM_READ:
		printf(" Error reading image.\n");
		break;
	case CT_IM_FITS_ERR:
		printf(" FITS Error.\n");
		break;
	default:
		printf(" Unknown Error.\n");
	}
	/* si llega hasta aca despliego... */
	if (good_image_flag) {
		caxpa_display(filename);
	}
	
	return 0;
}

#if 0
int com_get_old (char *arg)
{
	int len ;//= CCD_Y_LEN * CCD_X_LEN ;
	int *data;
	int status;
	char filename[_POSIX_PATH_MAX];
	struct image_2d im;
	time_t t;

	if ( cc_act_rcv_setup(cam, &im.s) < 0) {
		printf(" Error reading image setup.\n");
		return -1;
	}
	len = im.s.naxis2 * im.s.naxis1;
	if ( im.s.naxis2 > CCD_Y_LEN || im.s.naxis2 <= 0 ||  /* 384 */
	     im.s.naxis1 > CCD_X_LEN || im.s.naxis1 <= 0 ) { /* 576 */
		printf(" invalid dimension %d x %d data image\n",
		       im.s.naxis2, im.s.naxis1);
		return -1;
	}
	data = (int *) malloc(len * sizeof(int));
	printf(" geting %d x %d data image: ", im.s.naxis2, im.s.naxis1);
	if ( cc_act_rcv_data(cam, data, &len) < 0) {
		printf(" Error reading image.\n");
		free(data);
		return -1;
	}
	im.data = data;
	sprintf(im.comment,
		"stari camera control v %s, (c) Federico A. Bareilles (License GPL)", VERSION);
	
	if (arg[0] == 0) {
		time(&t);
		sprintf(filename,"im%10ld.fits", t); 
		
	}else{
		sprintf(filename,"%s", arg);
	}
	status = image_2d_save(filename, &im);
	free(data);
	if (status != 0) {
		printf("Fail.\n");
		fits_report_error(stdout, status);
		return -1;
	}

	printf("ok.\n");
	return 0;
}
#endif

int com_focus (char *arg)
{
	/* FIXME: debe ejecutarlo continuamante hasta que se toque una
           tecla */
	cc_act_focus(cam);
	return 0;
}


int com_shade (char *arg)
{
	cc_act_shade(cam);
	return 0;
}


int com_clear (char *arg)
{
	cc_act_clear(cam);
	return 0;
}


int com_nroi (char *arg)
{
	int v;

        if (arg[0] == 0) {
		cc_get_roi_index(cam, &v);
                printf(" current number of ROIs is %d\n", v);
                return 0;
        }

	v = atoi(arg);
	cc_set_roi_index(cam, v);

	return 0;
}

int com_roi_expose (char *arg)
{
  cc_act_roi_acquisition(cam);

  return 0;
}


int com_roi (char *arg)
{
	int xy[4], i;
	char *argp;

        if (arg[0] == 0) {
		cc_get_box_x0(cam, &xy[0]);
		cc_get_box_y0(cam, &xy[1]);
		cc_get_box_x_len(cam, &xy[2]);
		cc_get_box_y_len(cam, &xy[3]);
                printf(" current ROI is %d %d %d %d\n", 
		       xy[0], xy[1], xy[2], xy[3]);
		
                return 0;
        }

	for(i = 0; i < 4 && arg != NULL; i++) {
		xy[i] = (int) strtod(arg, &argp);
		arg = argp;
	}
	if ( i != 4 ) {
		printf(" roi: bad argument %s\n", arg);
		return -1;
	}
	cc_set_box_x0(cam, xy[0]);
	cc_set_box_y0(cam, xy[1]);
	cc_set_box_x_len(cam, xy[2]);
	cc_set_box_y_len(cam, xy[3]);
	cc_act_set_box(cam);
	return 0;
}


int com_box_x0 (char *arg)
{
	int v;

        if (arg[0] == 0) {
		cc_get_box_x0(cam, &v);
                printf(" X origin is %d\n", v);
                return 0;
        }

	v = atoi(arg);
	cc_set_box_x0(cam, v);

	return 0;
}


int com_box_y0 (char *arg)
{
	int v;

        if (arg[0] == 0) {
		cc_get_box_y0(cam, &v);
                printf(" Y origin  is %d\n", v);
                return 0;
        }

	v = atoi(arg);
	cc_set_box_y0(cam, v);

	return 0;
}


int com_box_x_len (char *arg)
{
	int v;

        if (arg[0] == 0) {
		cc_get_box_x_len(cam, &v);
                printf(" Length of X is %d\n", v);
                return 0;
        }

	v = atoi(arg);
	cc_set_box_x_len(cam, v);

	return 0;
}


int com_box_y_len (char *arg)
{
	int v;

        if (arg[0] == 0) {
		cc_get_box_y_len(cam, &v);
                printf(" Length of Y is %d\n", v);
                return 0;
        }

	v = atoi(arg);
	cc_set_box_y_len(cam, v);

	return 0;
}


int com_box_get (char *arg)
{
  cc_act_get_box(cam);

  return 0;
}


int com_box_set (char *arg)
{
  cc_act_set_box(cam);

  return 0;
}


int com_mouse_get (char *arg)
{
	struct mouse_pointer mp;

	cc_act_rcv_mouse_xyz(cam, &mp);
	printf(" x = %3d, y = %3d, val = %4d\n", mp.x, mp.y, mp.val);

	return 0;
}

int com_system (char *arg)
{
	printf("[%s]\n", arg);
	return system(arg);
}


int com_test (char *arg)
{
#if 0
	int b = atoi(arg);
	char str[20];
	cc_act_move_mouse(cam, b, b);
	
	get_int_from_cc(cam, "W", &b);
	printf("  Inten: %d\n", b);
	put_int_to_cc(cam,"V", 100);
	memset(str, 0, 20);
	
	if( b != 0) return 0;
	
	get_int_from_cc(cam, arg, &b);
	printf("  Result: %d\n", b);
#endif
	
	//my_exec("vim");
	
	return 0;
}


int main(int argc, char * const argv[])
{
	int op;

	verbose_flag = PERR;
	while  ((op = getopt(argc, argv, "v")) != EOF)
		switch( op ){
		case 'v':
			verbose_flag = (verbose_flag << 1) | 1;
			break;
		default:
			printf("Use %s -v (incremental)\n", argv[0]);
			exit(-1);
		}

	cam = open_camera();
	//cam = 0;

	if( cam >= 0 ) {
		caxpa_open(); /*FIXME camera_xpa.h */
		command_main();
		close_camera(cam);
		caxpa_close(); /* FIXME camera_xpa.h*/
	}else{
		PVERB(PERR," stari: can not connect to cc\n");
	}

        return 0;
}

