/** 
 *   Camera Control for STAR I.
 *
 *   (c) Copyright 2004 Federico Bareilles <fede@iar.unlp.edu.ar>,
 *   Instituto Argentino de Radio Astronomia (IAR).
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License as
 *   published by the Free Software Foundation; either version 2 of
 *   the License, or (at your option) any later version.
 *     
 *   The author does NOT admit liability nor provide warranty for any
 *   of this software. This material is provided "AS-IS" in the hope
 *   that it may be useful for others.
 *
 **/
#define _GNU_SOURCE

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>

#include <fitsio.h>
#include "fits_io.h"

int printerror( int status)
{
	/*****************************************************/
	/* Print out cfitsio error messages and exit program */
	/*****************************************************/

	if (status) {
		fits_report_error(stderr, status); /* print error report */
		return -1;
	}
	
	return 0;
}

static const char *str_im_type[] = {
	"Undefined", /* IMT_NONE */
	"OBJECT",    /* IMT_IMAGE */
	"DARK",      /* IMT_DARK */
	"BIAS",      /* IMT_BIAS */
	"FOCUS",     /* IMT_FOCUS */
	"Undefined", /* IMT_SHADE */
	"Undefined", /* IMT_CLEAR */
	"FLATT",     /* IMT_FLATT */
	""
};

int image_2d_save(char *filename, struct image_2d *im)
{
	char file[_POSIX_PATH_MAX];
	fitsfile *fptr;  /* pointer to the FITS file, defined in fitsio.h */
	int status;
	long fpixel, nelements;
	int bitpix   =  USHORT_IMG; /* 16-bit unsigned short pixel values */
	long naxis    =   2;        /* 2-dimensional image                */  
    	long naxes[2] = { im->s.naxis2, im->s.naxis1 }; 

	sprintf(file,"!%s", filename);	
	status = 0; /* initialize status before calling fitsio routines */
	if (fits_create_file(&fptr, file, &status))
		printerror( status );
	
	if ( fits_create_img(fptr,  bitpix, naxis, naxes, &status) )
		printerror( status );
	
	if ( fits_delete_key(fptr, "COMMENT", &status) )
		printerror( status );
	if ( fits_delete_key(fptr, "COMMENT", &status) )
		printerror( status );

	fpixel = 1;  /* first pixel to write      */
	nelements = naxes[0] * naxes[1]; /* number of pixels to write */

	/* write the array of unsigned integers to the FITS file */
	if ( fits_write_img(fptr, TUINT, fpixel, nelements, im->data, 
			    &status) )
		printerror( status );

	if ( fits_update_key(fptr, TSTRING, "IMAGETYP", 
			     str_im_type[im->s.im_type],
			     "Type of observation", &status) )
		printerror( status );
	
	if ( fits_update_key(fptr, TDOUBLE, "EXPOSURE", &im->s.exposure,
			     "Total Exposure Time", &status) )
		printerror( status );

	if ( fits_update_key(fptr, TINT, "GAIN", &im->s.gain,
			     "System gain factor 1 or 4", &status) )
		printerror( status );

	if ( fits_update_key(fptr, TINT, "TEMPERAT", &im->s.temperature,
			     "CCD Temperature", &status) )
		printerror( status );

	if ( fits_write_comment(fptr, im->comment, &status) )
		printerror( status );
	
	if ( fits_close_file(fptr, &status) ) /* close the file */
		printerror( status );
	
	return status;
}
