/** 
 *   Camera Control for STAR I.
 *
 *   (c) Copyright 2004 Federico Bareilles <fede@iar.unlp.edu.ar>,
 *   Instituto Argentino de Radio Astronomia (IAR).
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License as
 *   published by the Free Software Foundation; either version 2 of
 *   the License, or (at your option) any later version.
 *     
 *   The author does NOT admit liability nor provide warranty for any
 *   of this software. This material is provided "AS-IS" in the hope
 *   that it may be useful for others.
 *
 **/
#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "iar_bus.h"
#define USE_IAR_BUS
#  include "camera.h"
#include "camera_ib_pk.h"
#include "my_stdio.h"



static ssize_t soft_read_time_out(struct timeval tv, int fd, 
				  void *buf, size_t count)
{
        fd_set rfds;
        int fd_max = fd + 1;
        int retval;

        FD_ZERO(&rfds);
        FD_SET(fd, &rfds);
        retval = select(fd_max, &rfds, NULL, NULL, &tv);
        if (retval) {
                if( FD_ISSET(fd, &rfds) ) {
                        retval = read(fd, buf, count);
#ifdef DEBUG_MODE
                        if (verbose_flag & PINFO) {
                                int __i;
                                PVERB(PINFO, 
				      "# camera_io_ib_client.c read <- ");
                                for(__i=0;__i<retval;__i++)
                                        PVERB(PINFO, "0x%02x ",
                                              *(((u_char *)buf)+__i) );
                                PVERB(PINFO, "\n");
                        }
#endif
                        return retval;
                }
        }

        return -1;
}


ssize_t soft_read(int fd, void *buf, size_t count)
{
        struct timeval tv;

        tv.tv_sec = 10;
        tv.tv_usec = 200000;

        return soft_read_time_out(tv, fd, buf, count);
}


int my_ibus_read(int fd, struct ib_packet *d)
{
	int ret;

        ret = soft_read(fd, d, sizeof(struct ib_packet_head) );
        if( d->h.size && ret > 0){
                ret += soft_read(fd, d->data, d->h.size );
        }

        return ret;
}



int get_int_from_cc(int dev, char *cmd, int *val)
{
	struct ib_packet pk;
	struct cc_packet *ccp;
	int ret;
	int l;

	ccp = (struct cc_packet *)&pk.data;
	l = sizeof(struct cc_packet_head);
	l += sprintf(ccp->data,"%s", cmd);
	ccp->h.cmd = CC_GET_INT;
	ccp->h.size = 0;
	ibus_send_data_to(dev, CC_IB_ID, ccp, l);

	ret = my_ibus_read(dev, &pk);
	if( ret > 0) {
		*val = atoi(ccp->data);
	}

	return ret;		
}


int get_str_from_cc(int dev, char *cmd, char *str, int len)
{
	struct ib_packet pk;
	struct cc_packet *ccp;
	int ret;
	int l;

	ccp = (struct cc_packet *)&pk.data;
	l = sizeof(struct cc_packet_head);
	l += sprintf(ccp->data,"%s", cmd);
	ccp->h.cmd = CC_GET_STR;
	ccp->h.size = len;
	ibus_send_data_to(dev, CC_IB_ID, ccp, l);

	ret = my_ibus_read(dev, &pk);
	if( ret > 0) {
		memcpy(str, ccp->data, len);
	}

	return ret;

}


int put_int_to_cc(int dev, char *cmd, int val)
{
	struct ib_packet pk;
	struct cc_packet *ccp;
	int l;

	ccp = (struct cc_packet *)&pk.data;
	l = sizeof(struct cc_packet_head);
	l += sprintf(ccp->data,"%c%d", cmd[0], val);
	ccp->h.cmd = CC_PUT_INT;
	ccp->h.size = 0;
	return ibus_send_data_to(dev, CC_IB_ID, ccp, l);
}


int put_str_to_cc(int dev, char *cmd, char *str)
{
	struct ib_packet pk;
	struct cc_packet *ccp;
	int l;

	ccp = (struct cc_packet *)&pk.data;
	l = sizeof(struct cc_packet_head);
	l += sprintf(ccp->data,"%c%s", cmd[0], str);
	ccp->h.cmd = CC_PUT_STR;
	ccp->h.size = 0;
	return ibus_send_data_to(dev, CC_IB_ID, ccp, l);
}


int action_to_cc(int dev, char *cmd)
{
	struct ib_packet pk;
	struct cc_packet *ccp;
	int l;

	ccp = (struct cc_packet *)&pk.data;
	l = sizeof(struct cc_packet_head);

	//l += sprintf(ccp->data,"%s", cmd); /* FIXME */
	l++;
	ccp->data[0] =  cmd[0];
	ccp->data[1] = 0;

	ccp->h.cmd = CC_ACTION;
	ccp->h.size = 0;

	return ibus_send_data_to(dev, CC_IB_ID, ccp, l);

}


int send_data_to_cc(int dev, char *data, int len)
{
	struct ib_packet pk;
	struct cc_packet *ccp;
	int l;

	ccp = (struct cc_packet *)&pk.data;
	l = sizeof(struct cc_packet_head);
	memcpy(ccp->data, data, len);
	l += len;
	ccp->h.cmd = CC_SEND_DATA;
	ccp->h.size = len;

	if ( ibus_send_data_to(dev, CC_IB_ID, ccp, l) > 0) {
		return len;
	}else{
		return -1;
	}
}


int rcv_data_from_cc(int dev, char *data, int *len)
{
	struct ib_packet pk;
	struct cc_packet *ccp;
	int ret, rlen;
	int l;

	ccp = (struct cc_packet *)&pk.data;
	l = sizeof(struct cc_packet_head);
	ccp->h.cmd = CC_RCV_DATA;
	ccp->h.size = *len;
	ibus_send_data_to(dev, CC_IB_ID, ccp, l);

	rlen = 0;
	while (rlen < *len) {
		ret = my_ibus_read(dev, &pk);
		if( ret < 0 ) {
			PVERB(PERR,"rcv_data_client: ERROR %d\n", ret);
			return -1;
		}
		ret -=  sizeof(struct ib_packet_head) + 
			sizeof(struct cc_packet_head);
		PVERB(PINFO,"rcv_data_client: %d\n", ret);
		if( ret > 0) {
			memcpy(data+rlen, ccp->data, ret);
			rlen += ret;
		}else{
			break;
		}
	}
	*len = rlen;

	return rlen;
}


int rcv_setup_from_cc(int dev, struct camera_setup *set)
{
	struct ib_packet pk;
	struct cc_packet *ccp;
	int ret;
	int l;

	ccp = (struct cc_packet *)&pk.data;
	l = sizeof(struct cc_packet_head);
	ccp->h.cmd = CC_RCV_SETUP;
	ccp->h.size = sizeof(struct camera_setup);
	ibus_send_data_to(dev, CC_IB_ID, ccp, l);

	ret = my_ibus_read(dev, &pk);
	if( ret > 0) {
		memcpy(set, ccp->data, sizeof(struct camera_setup));
	}

	return ret;
}


int rcv_mouse_from_cc(int dev, struct mouse_pointer *mp)
{
	struct ib_packet pk;
	struct cc_packet *ccp;
	int ret;
	int l;

	ccp = (struct cc_packet *)&pk.data;
	l = sizeof(struct cc_packet_head);
	ccp->h.cmd = CC_RCV_MOUSE;
	ccp->h.size = sizeof(struct mouse_pointer);
	ibus_send_data_to(dev, CC_IB_ID, ccp, l);

	ret = my_ibus_read(dev, &pk);
	if( ret > 0) {
		memcpy(mp, ccp->data, sizeof(struct mouse_pointer));
	}

	return ret;
}

