/** 
 *   Camera Control for STAR I.
 *
 *   (c) Copyright 2004 Federico Bareilles <fede@iar.unlp.edu.ar>,
 *   Instituto Argentino de Radio Astronomia (IAR).
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License as
 *   published by the Free Software Foundation; either version 2 of
 *   the License, or (at your option) any later version.
 *     
 *   The author does NOT admit liability nor provide warranty for any
 *   of this software. This material is provided "AS-IS" in the hope
 *   that it may be useful for others.
 *
 **/
#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <stdarg.h>
#include <unistd.h>
#include <malloc.h>
#include <gpib/ib.h>
#include <math.h>
#include <sys/time.h>
#define HW_GPIB
#include "camera.h"
#include "my_stdio.h"
#include "camera_setup.h"


static struct timeval expiration = {
	.tv_sec  = 0,  /* seconds */
	.tv_usec = 0,  /* microseconds */
};



static int my_ibwrite_generic(
	int (*write_func)(int ud, const void *data, long num_bytes),
	int dev, const char *fmt, ...)
{
	va_list ap;
	int l = -1;
	int n, size = 80;
	char *buff = NULL;
	struct timeval now;

	if ((buff = malloc (size)) == NULL) return l;

	while (1) {
		/* Try to print in the allocated space. */
		va_start(ap, fmt);
		//memset(buff, 0, size);
		n = vsnprintf (buff, size, fmt, ap);
		va_end(ap);
		if (n > -1 && n < size) {
			break;
		}
		else {
			/* Else try again with more space. */
			if (n > -1)    /* glibc 2.1 */
				size = n+1; /* precisely what is needed */
			else           /* glibc 2.0 */
				size *= 2;  /* twice the old size */
			if ((buff = realloc (buff, size)) == NULL)
				return l;
		}
	}
	if (buff != NULL) {
		/* Wait until integration expired (if) */
		gettimeofday(&now, NULL);
		while (timercmp(&expiration, &now, >)) {
			PVERB(PINFO,"# Waiting end of integration.\n");
			sleep(expiration.tv_sec - now.tv_sec);
			gettimeofday(&now, NULL);
		}
		timerclear(&expiration);
		buff[n] =0;
		PVERB(PINFO,"# Send: %d %s\n", n, buff);
#ifdef DEMO_MODE
		l = write(dev, buff, n);
		write(dev, "\n", 1);

#else
		l = write_func(dev, buff, n);
#endif

#ifdef VERBOSE_MODE
		if (l & ERR /*&& !(l & TIMO)*/) {
			SYSLOG(LOG_ERR,
			       "Error sending: %s [ibsta: 0x%x][iberr: 0x%x]\n",
			      buff, l, iberr);
		}
#endif
		free(buff);
	}

	return l;
}


/* Wait for service request */
static int cc_wsq(int dev)
{
	u_char spr;

	while (!(ibrsp(dev, &spr) & ERR)) {
#ifdef DEBUG_MODE
	  if(spr)
	    PVERB(PINFO," Service request ret: 0x%x\n", spr);
#endif
		if (spr == CMD_OK) {
			break;
		}else if (spr == CMD_ERROR) {
			return -1;
		}
	}
#ifdef NI_NAT4882_ISA
	usleep(WSR_USLEEP);
#endif
	return 0;
}


static int __get_int_from_cc(int dev, char *cmd, int *val)
{
	char ctmp[RW_BUF_SIZE];

	if ( my_ibwrite_generic(ibwrt, dev, "@%c", *cmd) & ERR) 
		return -1;
	if (cc_wsq(dev) != 0) 
		return -1;
	memset(ctmp, 0, RW_BUF_SIZE);
#ifndef DEMO_MODE
	if (ibrd(dev, ctmp, RW_BUF_SIZE) & ERR) {
		return -1;
	}else{
		*val = atoi(ctmp);

	}		
#endif
	return 0;
}

int get_int_from_cc(int dev, char *cmd, int *val)
{
	int v = -1;
	int ret; 
	int retry = 0;

	do {
		*val = v;
		ret = __get_int_from_cc(dev, cmd, &v);
		printf("Val: %d, ret = %d, retry = %d\n", v, ret, retry);
	}while( (v != *val) && (ret == 0) && retry++ < CC_RETRY_GET);
	
	return ret;
}

int get_str_from_cc(int dev, char *cmd, char *str, int len)
{
	if ( my_ibwrite_generic(ibwrt, dev, "@%s", cmd) & ERR) 
		return -1;
	if (cc_wsq(dev) != 0) 
		return -1;
	memset(str, 0, len);
	ibtmo(dev, T1s);
	if (ibrd(dev, str, len) & ERR) {
		str = NULL;
		ibtmo(dev, CC_TIME_OUT);
		return -1;
	}
	PVERB(PINFO,"get_str_from_cc: done.\n");
	ibtmo(dev, CC_TIME_OUT);
	
	return 0;
}


int put_int_to_cc(int dev, char *cmd, int val)
{
	if ( my_ibwrite_generic(ibwrt, dev, "!%c%d", *cmd, val) & ERR) 
		return -1;
	if (cc_wsq(dev) != 0) 
		return -1;

	return 0;
}


int put_str_to_cc(int dev, char *cmd, char *str)
{
	if ( my_ibwrite_generic(ibwrt, dev, "!%c%s", *cmd, str) & ERR)
		return -1;
	if (cc_wsq(dev) != 0) 
		return -1;

	return 0;
}


int action_to_cc(int dev, char *cmd)
{
	int im_type = IMT_NONE;
	int wait = 0;

	switch(cmd[0]) {
	case 'E': /* Trigger */
	case 'A': /* Exposure integration*/
		im_type = IMT_IMAGE;
		wait = 1;
		break;
	case 'B': /* Dark integration */
		im_type = IMT_DARK;
		wait = 1;
		break;
	case 'C': /* Bias integration */
		im_type = IMT_BIAS;
		break;
	case 'D': /* Focus */
	case 'W': /* Focus :-) */
		im_type = IMT_FOCUS;
		wait = 1;
		break;
	case 'G': /* Shade */
		im_type = IMT_SHADE;
		break;
	case 'H': /* Clear */
		im_type = IMT_CLEAR;
		break;
	default:
		break;
	}

	if(im_type != IMT_NONE) {
	  if (camera_get_setup(dev, im_type) != 0) {
	    return -1;
	  }
	}

	gettimeofday(&expiration, NULL);
	/* FIXME: %c or %s mmm?? */
	if ( my_ibwrite_generic(ibwrt, dev, ":%c", cmd[0]) & ERR) 
		return -1;
	if (cc_wsq(dev) != 0) 
		return -1;

	if (wait) {
		struct camera_setup cs;
		float t;
		camera_current_setup(dev, &cs);
		t = cs.exposure + 
			(double) expiration.tv_sec +
			(double) (expiration.tv_usec) / 1000000.0;
		expiration.tv_sec = (long) t;
		expiration.tv_usec = (long) 
			((t - (double) expiration.tv_sec) * 1000000.0);
	}

	return 0;
}


int send_data_to_cc(int dev, char *data, int len)
{
	ibtmo(dev, T1s);
	ibwrt(dev, data, len);
	ibtmo(dev, CC_TIME_OUT);

	return ibcnt >> 1;
}


int rcv_data_from_cc(int dev, char *data, int *len)
{
	int rcv;
#ifdef DEMO_MODE
	int i,j=0;

	for(i=0; i < (*len>>1);i++){
		if (j == 384) j = 0;
		data[i<<1] = j & 0xff;
		data[i<<1|1] = (j & 0xf00) >> 8;
		j++;
	}
#else
	ibtmo(dev, T10s);
	ibrd(dev, data, *len);
	rcv = ibcnt;
	ibtmo(dev, CC_TIME_OUT);
	*len = rcv;
#endif
	rcv = *len;

	return rcv;
}


/** 
 * Las siguientes funciones estan implementedas del lado GPIB,
 * solamente para tener mayor eficiencia en al caso de usar la
 * aplicacion separada.
 **/


int rcv_setup_from_cc(int dev, struct camera_setup *set)
{
	return camera_current_setup(dev, set);
}


int rcv_mouse_from_cc(int dev, struct mouse_pointer *mp)
{
	int ret;
#ifdef DEMO_MODE
	mp->x = 100;
	mp->y = 100;
	mp->val = 1500;
	ret = 0;
#else
	if( (ret = get_int_from_cc(dev, "V", &mp->x)) < 0) return ret; 
	if( (ret = get_int_from_cc(dev, "U", &mp->y)) < 0) return ret; 
	ret = get_int_from_cc(dev, "W", &mp->val);
#endif
	return ret; 
}
