/** 
 *   Camera Control for STAR I.
 *
 *   (c) Copyright 2004 Federico Bareilles <fede@iar.unlp.edu.ar>,
 *   Instituto Argentino de Radio Astronomia (IAR).
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License as
 *   published by the Free Software Foundation; either version 2 of
 *   the License, or (at your option) any later version.
 *     
 *   The author does NOT admit liability nor provide warranty for any
 *   of this software. This material is provided "AS-IS" in the hope
 *   that it may be useful for others.
 *
 **/
#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <stdarg.h>
#include <unistd.h>
#include <malloc.h>
#include "camera.h"
#include "camera_header.h"
#include "camera_setup.h"
#include "my_stdio.h"


#define SWAP_BYTE(b) b = (((b << 8) & 0xff00) | ((b >> 8) & 0x00ff))

/******************************************************************************
 * Camera controller commands:
 *
 **/

int cc_get_exposure_time(int dev, int *time)
{
	return get_int_from_cc(dev, "A", time);
}
		

int cc_set_exposure_time(int dev, int time)
{
	return put_int_to_cc(dev, "A", time);
}


int cc_get_gain(int dev, int *gain)
{
	return get_int_from_cc(dev, "B", gain);
}

		
int cc_set_gain(int dev, int gain)
{
	if (gain == GAIN_HIGH || gain == GAIN_LOW)
		return put_int_to_cc(dev, "B", gain);
	else
		return -1;
}


int cc_get_temp_regulation(int dev, int *temp)
{
	return get_int_from_cc(dev, "C", temp);
}

		
int cc_set_temp_regulation(int dev, int temp)
{
	if (temp == TEMP_REG_ON || temp == TEMP_REG_OFF)
		return put_int_to_cc(dev, "C", temp);
	else
		return -1;
}


int cc_get_box_y0(int dev, int *y)
{
	return get_int_from_cc(dev, "D", y);
}
		

int cc_set_box_y0(int dev, int y)
{
	return put_int_to_cc(dev, "D", y);
}


int cc_get_box_x0(int dev, int *x)
{
	return get_int_from_cc(dev, "E", x);
}
		

int cc_set_box_x0(int dev, int x)
{
	return put_int_to_cc(dev, "E", x);
}


int cc_get_box_y_len(int dev, int *y)
{
	return get_int_from_cc(dev, "F", y);
}
		

int cc_set_box_y_len(int dev, int y)
{
	return put_int_to_cc(dev, "F", y);
}


int cc_get_box_x_len(int dev, int *x)
{
	return get_int_from_cc(dev, "G", x);
}
		

int cc_set_box_x_len(int dev, int x)
{
	return put_int_to_cc(dev, "G", x);
}


int cc_get_zoom_factor(int dev, int *z)
{
	return get_int_from_cc(dev, "H", z);
}
		

int cc_set_zoom_factor(int dev, int z)
{
	return put_int_to_cc(dev, "H", z);
}


int cc_get_auto_win(int dev, int *w)
{
	return get_int_from_cc(dev, "I", w);
}

		
/* This function turns the automatic window calculation function off
   and on */
int cc_set_auto_win(int dev, int w)
{
	return put_int_to_cc(dev, "I", w);
}


int cc_get_date_time(int dev, time_t *time)
{
	PVERB(PWARN,"cc_get_date_time: not implemented\n");
	return -1; /* FIXME */
}
		

int cc_set_date_time(int dev, time_t time)
{
	struct tm *t;
	char date_str[8];

	t = localtime(&time);
	sprintf(date_str,"%02d%02d%02d", 
		t->tm_mon + 1, t->tm_mday, t->tm_year + 1900 - 2000); 
	if (put_str_to_cc(dev, "L", date_str) != 0)
		return -1;

	sprintf(date_str,"%02d%02d%02d", t->tm_hour, t->tm_min, t->tm_sec);
	if (put_str_to_cc(dev, "M", date_str) != 0)
		return -1;

	return 0;
}


int cc_get_win_max(int dev, int *w)
{
	return get_int_from_cc(dev, "N", w);
}


int cc_set_win_max(int dev, int w)
{
	return put_int_to_cc(dev, "N", w);
}


int cc_get_win_min(int dev, int *w)
{
	return get_int_from_cc(dev, "O", w);
}


int cc_set_win_min(int dev, int w)
{
	return put_int_to_cc(dev, "O", w);
}

#if 0
static int cc_header_swap_byte(struct cc_image_head *h)
{
	SWAP_BYTE(h->exposure);
	SWAP_BYTE(h->x0);
	SWAP_BYTE(h->y0);
	SWAP_BYTE(h->x_len);
	SWAP_BYTE(h->y_len);
	SWAP_BYTE(h->temperature);
	SWAP_BYTE(h->gain);

	return 0;
}
#endif

int cc_get_cc_header(int dev, struct cc_image_head *h)
{
	int ret;

	memset(h, 0, sizeof(struct cc_image_head));
	ret = get_str_from_cc(dev, "P", (char *) h, 
			      sizeof(struct cc_image_head));
#if 0
	if (ret == 0) {
		printf("# comment %s\n", h->comment);
		printf("# time_date %s\n", h->time_date);
		printf("# x0 %d\n", h->x0);
		printf("# y0 %d\n", h->y0);
		printf("# x_len %d\n", h->x_len);
		printf("# y_len %d\n", h->y_len);
		printf("# temperature %d\n", h->temperature);
		printf("# gain %d\n", h->gain);
		//cc_header_swap_byte(h);
	}
#endif
	return ret;
}


int cc_set_cc_header(int dev, struct cc_image_head h)
{
	/* cc_header_swap_byte(&h); */
	PVERB(PWARN,"cc_set_cc_header: not implemented\n");
	return -1; /* FIXME */
}


int cc_get_binning(int dev, int *b)
{
        return get_int_from_cc(dev, "Q", b);
}


int cc_set_binning(int dev, int b)
{
	return put_int_to_cc(dev, "Q", b);
}


int cc_get_roi_index(int dev, int *i)
{
	return get_int_from_cc(dev, "R", i);
}


int cc_set_roi_index(int dev, int i)
{
	return put_int_to_cc(dev, "R", i);
}


int cc_get_roi_delay(int dev, int *d)
{
	return get_int_from_cc(dev, "S", d);
}


int cc_set_roi_delay(int dev, int d)
{
	return put_int_to_cc(dev, "S", d);
}


int cc_get_temperature(int dev, int *t)
{
	return get_int_from_cc(dev, "T", t);
}

/******************************************************************************
 * Camera controller action commands:
 *
 **/


int cc_act_exposure(int dev)
{
	return action_to_cc(dev, "A");
}


int cc_act_dark(int dev)
{
	return action_to_cc(dev, "B");
}


int cc_act_bias(int dev)
{
	return action_to_cc(dev, "C");
}


int cc_act_focus(int dev)
{
	/**
	   D is real command for focus action in the manual, but never
	   return from this :-). W is "calibration command, and is very
	   useful for focuses
	**/
	/* return action_to_cc(dev, "D"); */
	return action_to_cc(dev, "W");
}


int cc_act_triger(int dev)
{
	return action_to_cc(dev, "E");
}


int cc_act_shade(int dev)
{
	return action_to_cc(dev, "G");
}


int cc_act_clear(int dev)
{
	return action_to_cc(dev, "H");
}


int cc_act_create_lut(int dev)
{
	return action_to_cc(dev, "R");
}


int cc_act_send_data(int dev, int *data, int len)
{
	int i;
	char data2cc[len << 1];

	for(i=0; i < len; i++) {
		data2cc[(i<<1)]   = data[i] & 0xff;
		data2cc[(i<<1)|1] = data[i] & 0x0f00 >> 8;
	}

	if (action_to_cc(dev, "I") != 0) {
		PVERB(PERR,"cc_act_send_data: error\n");
		return -1;
	}
	return send_data_to_cc(dev, data2cc, len << 1);
}


int cc_act_rcv_data(int dev, int *data, int *len)
{
	int i;
	int l;
	char data2cc[*len << 1];

	if (action_to_cc(dev, "J") != 0) {
		PVERB(PERR,"cc_act_rcv_data: error\n");
		l = *len << 1;
		rcv_data_from_cc(dev, data2cc, &l);
		*len = -1;
		return -1;
	}
	sleep(1);

	l = *len << 1;
	if( rcv_data_from_cc(dev, data2cc, &l) < 0) {
		return -1;
	}
	*len = l >> 1;
	for(i=0; i < *len; i++) {
		data[i] = (data2cc[(i<<1)] & 0xff) | (
			(data2cc[(i<<1)|1] & 0x0f) << 8);
	}

	return *len;

}


int cc_act_rcv_setup(int dev, struct camera_setup *set)
{
	return rcv_setup_from_cc(dev, set);
}


int cc_act_rcv_mouse_xyz(int dev, struct mouse_pointer *mp)
{
	return rcv_mouse_from_cc(dev, mp);
}


int cc_act_set_box(int dev)
{
	return action_to_cc(dev, "S");
}


int cc_act_get_box(int dev)
{
	return action_to_cc(dev, "T");
}


int cc_act_send_lut(int dev, char *lut)
{
	lut[0] = 0;
	lut[4095] = 255;

	if (action_to_cc(dev, "U") != 0) {
		PVERB(PERR,"cc_act_send_lut: error\n");
		return -1;
	}

	return send_data_to_cc(dev, lut, 4096);
}


int cc_act_roi_acquisition(int dev)
{
	return action_to_cc(dev, "X");
}


int cc_act_move_mouse(int dev, int x_delta, int y_delta)
{
	char s[12];

	sprintf(s, "V%d:%d", x_delta, y_delta);

	return action_to_cc(dev, s);
}


